<?php
/*--------------------------------------------------------------------------------------------------
    OnGetSelectedQuantityEventListener.php 2020-03-18
    Gambio GmbH
    http://www.gambio.de
    Copyright (c) 2016 Gambio GmbH
    Released under the GNU General Public License (Version 2)
    [http://www.gnu.org/licenses/gpl-2.0.html]
    --------------------------------------------------------------------------------------------------
 */


namespace Gambio\Shop\Stock\SellingUnitQuantity\Database\Listeners;

use Gambio\Shop\SellingUnit\Database\Unit\Events\Interfaces\OnGetSelectedQuantityEventInterface;
use Gambio\Shop\SellingUnit\Unit\Exceptions\InsufficientQuantityException;
use Gambio\Shop\SellingUnit\Unit\Exceptions\InvalidQuantityGranularityException;
use Gambio\Shop\SellingUnit\Unit\ValueObjects\Interfaces\IncludeReservedOnQuantityInterface;
use Gambio\Shop\SellingUnit\Unit\ValueObjects\SellingUnitId;

/**
 * Class OnGetSelectedQuantityListener
 */
class OnGetSelectedQuantityEventListener
{

    /**
     * Product Type Download
     */
    private const DOWNLOAD_TYPE = 2;
    /**
     * Default Type(Products)
     */
    private const DEFAULT_TYPE = 1;


    /**
     * @var bool
     */
    protected $downloadStockCheck;
    /**
     * @var bool
     */
    protected $stockCheck;
    /**
     * @var float|double
     */
    private $maxProductQuantity;


    /**
     * OnGetSelectedQuantityListener constructor.
     *
     * @param float $maxProductQuantity
     * @param bool $downloadStockCheck
     * @param bool $stockCheck
     * Example
     * ***ProductInfoThemeContentView***
     * = AvailableQuantity (All the quantity in the stock - Product, Combination or Attributes)
     *      100 Products
     *      200 Combinations
     *      20 attribute
     *      ================================
     *      SellingUnitAvailableQuantity = 20
     *      ================================
     * = ReservedQuantity (All the quantity already in the shopping cart)
     *     10 quantity
     * = RequestedQuantity (Information supplied By User - the amount he want to buy)
     *      10 = OK
     *      11 = InsufficientQuantityException

     * ****ShoppingCart****
     * = AvailableQuantity (All the quantity in the stock - Product, Combination or Attributes)
     *      100 Products
     *      200 Combinations
     *      20 attribute
     *      ================================
     *      SellingUnitAvailableQuantity = 20
     *      ================================
     * = ReservedQuantity (All the quantity already in the shopping cart)
     *     10 quantity
     * = RequestedQuantity (Information supplied By User - the amount he want to buy)
     *      10 = OK
     *      11 = OK
     *      20 = OK
     *      21 = InsufficientQuantityException
     *
     */
    public function __construct(float $maxProductQuantity, bool $stockCheck, bool $downloadStockCheck)
    {

        $this->maxProductQuantity = $maxProductQuantity;
        $this->downloadStockCheck = $downloadStockCheck;
        $this->stockCheck = $stockCheck;
    }


    /**
     * @param OnGetSelectedQuantityEventInterface $event
     *
     * @return OnGetSelectedQuantityEventInterface
     */
    public function __invoke(OnGetSelectedQuantityEventInterface $event): OnGetSelectedQuantityEventInterface
    {

        if ($this->downloadStockCheck && $event->product()->getProductType() === static::DOWNLOAD_TYPE) {
            $quantity = $event->stock()->availableQuantity()->value();
            if ($event->requestedQuantity() instanceof IncludeReservedOnQuantityInterface) {
                $quantity += $event->stock()->reservedQuantity()->value();
            }
            if ($quantity < $event->requestedQuantity()) {
                $event->requestedQuantity()->stackException(new InsufficientQuantityException($event->id()->productId()->value(),
                    $event->stock()->availableQuantity()->value(), $event->requestedQuantity()->exception()));

            }
        } elseif ($this->stockCheck && $event->product()->getProductType() === static::DEFAULT_TYPE) {

            $quantity = $event->requestedQuantity()->value();

            //if there is some combination for the selected values
            if ($event->requestedQuantity() instanceof IncludeReservedOnQuantityInterface) {
                    $quantity -=   $event->stock()->reservedQuantity()->value();
            }

            if ($quantity > $event->stock()->availableQuantity()->value()) {
                $event->requestedQuantity()->stackException(new InsufficientQuantityException($event->id()->productId()->value(),
                    $event->stock()->availableQuantity()->value(), $event->requestedQuantity()->exception()));
            }
        }
        return $event;
    }

    /**
     * @param SellingUnitId $id
     * @return float
     */
    protected function getReservedQuantity(SellingUnitId $id): float
    {
        $result = 0;
        if (isset($_SESSION['cart']) && is_object($_SESSION['cart'])) {

            $result = (float)$_SESSION['cart']->get_quantity($id->productId()->value());
        }

        return $result;
    }
}

